/**
 * \file: mspin_connection_tcp_listener.c
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * mySPIN TCP/TLS Listener
 *
 * \component: MSPIN
 *
 * \author: Thilo Bjoern Fickel BSOT/PJ-ES1 thilo.fickel@bosch-softtec.com
 *
 * @copyright: (c) 2016 Bosch SoftTec GmbH
 *
 * \history
 * 0.1 TFickel Initial version
 *
 ***********************************************************************/

#include "mspin_connection_tcp_listener.h"
#include "mspin_connection_tcp_manager.h"
#include "mspin_connection_tcp_server.h"
#include "mspin_logging.h"

#include <errno.h>          //errno
#include <arpa/inet.h>      //inet_ntop, AF_INET
#include <sys/prctl.h>      //prctl
#include <netinet/tcp.h>    //SOL_TCP, TCP_*

#define MSPIN_TCP_LISTEN_BACKLOG 5
#define MSPIN_TCP_SOCKET_ADDR_REUSE
#define MSPIN_TCP_USE_KEEPALIVE
#define MSPIN_TCP_LISTEN_THREAD_TIMEOUT_MS 300

mspin_tcp_ListenerContext_t *gpTCPListenerContext = NULL;

#ifdef MSPIN_TCP_USE_KEEPALIVE
static void mspin_tcp_enableKeepAlive(int fd)
{
    int optval = 0;
    socklen_t optlen = sizeof(optval);

    //Check the status for the keep-alive option
    if (getsockopt(fd, SOL_SOCKET, SO_KEEPALIVE, &optval, &optlen) < 0)
    {
       mspin_log_printLn(eMspinVerbosityError, "%s(p) ERROR: getsockopt() failed",
               __FUNCTION__);
    }

    if (!optval)
    {
        //Set the option state
        optval = 1;
        optlen = sizeof(optval);
        if(setsockopt(fd, SOL_SOCKET, SO_KEEPALIVE, &optval, optlen) < 0)
        {
            mspin_log_printLn(eMspinVerbosityError,
                    "%s() ERROR: setsockopt() failed for SO_KEEPALIVE",
                    __FUNCTION__);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s() SO_KEEPALIVE enabled",
                    __FUNCTION__, (optval ? "ON" : "OFF"));
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s() SO_KEEPALIVE was already enabled",
                __FUNCTION__);
    }

    //Check the status for the keep-alive option after setting
    if (getsockopt(fd, SOL_SOCKET, SO_KEEPALIVE, &optval, &optlen) < 0)
    {
       mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: getsockopt() failed",
               __FUNCTION__);
    }
    else
    {
        if (optval)
        {
            optval = 1; //one probe is required
            optlen = sizeof(optval);
            if (setsockopt(fd, SOL_TCP, TCP_KEEPCNT, &optval, optlen) < 0)
            {
                mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: setsockopt() failed for 'TCP_KEEPCNT'",
                        __FUNCTION__);
            }

            optval = 1; //start after 1 second
            if (setsockopt(fd, SOL_TCP, TCP_KEEPIDLE, &optval, optlen) < 0)
            {
                mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: setsockopt() failed for 'TCP_KEEPIDLE'",
                        __FUNCTION__);
            }

            optval = 2; //every 2 seconds
            if (setsockopt(fd, SOL_TCP, TCP_KEEPINTVL, &optval, optlen) < 0)
            {
                mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: setsockopt() failed for 'TCP_KEEPINTVL'",
                        __FUNCTION__);
            }

            //TPlate: keepalive fails (no reaction, no ETIMEDOUT), if data is sent/tried to be sent
            // on the socket during the keepalive interval.
            // To detect this and other missing ACK, we use TCP_USER_TIMEOUT
            optval = 5000; //5000ms
            if (setsockopt(fd, SOL_TCP, TCP_USER_TIMEOUT, &optval, optlen) < 0)
            {
                mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: setsockopt() failed for 'TCP_USER_TIMEOUT'",
                        __FUNCTION__);
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: SO_KEEPALIVE is still %s",
                    __FUNCTION__, (optval ? "ON" : "OFF"));
        }
    }
}
#endif //MSPIN_TCP_USE_KEEPALIVE

static int mspin_tcp_accept(mspin_tcp_ListenerContext_t *pListener)
{
    struct sockaddr_in cli_addr = {0};
    socklen_t clilen = sizeof(struct sockaddr_in);
    int socketFD = -1;
    fd_set fds = {{0}};
    struct timeval tv;
    int selectResult = 0;
    int highestFD = 0;
    MSPIN_ERROR result = MSPIN_SUCCESS;

    if (!pListener)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(listener=%p) FATAL ERROR: Context is NULL",
                __FUNCTION__, pListener);
        return socketFD;
    }

    while (gpTCPListenerContext && !gpTCPListenerContext->quitListener && (0 == selectResult))
    {
        /* PRQA: Lint Message 529: Warning due symbol not subsequently referenced in the asm section. Not an issue */
        /*lint -save -e529*/
        FD_ZERO(&fds);
        /*lint -restore*/

        FD_SET(pListener->listenFD, &fds);
        highestFD = pListener->listenFD;

        tv.tv_sec = MSPIN_TCP_LISTEN_THREAD_TIMEOUT_MS/1000; //seconds
        tv.tv_usec = (MSPIN_TCP_LISTEN_THREAD_TIMEOUT_MS%1000)*1000; //milliseconds

        selectResult = select(highestFD+1, &fds, NULL, NULL, &tv);

        if (-1 == selectResult)
        {
            //Error => leave loop (selectResult is not 0)
            mspin_log_printLn(eMspinVerbosityError, "%s(listener=%p) ERROR: Select failed with '%s' (%d) => leave loop",
                    __FUNCTION__, pListener, strerror(errno), errno);
        }
        else if (0 == selectResult)
        {
            //Timeout => continue
            mspin_log_printLn(eMspinVerbosityVerbose, "%s(listener=%p) select returned 0 (=timeout) -> continue",
                    __FUNCTION__, pListener);
        }
        else
        {
            /* PRQA: Lint Message 64: accept type mismatch */
             /*lint -save -e64*/
             socketFD = accept(pListener->listenFD, (struct sockaddr*)&cli_addr, &clilen);
             /*lint -restore*/

             if (-1 == socketFD)
             {
                 mspin_log_printLn(eMspinVerbosityError, "%s(listener=%p) ERROR: accept failed with '%s'(%d)",
                         __FUNCTION__, pListener, strerror(errno), errno);
             }
             else
             {
                 char* ipAddr = inet_ntoa(cli_addr.sin_addr);
                 if (!pListener->quitListener)
                 {
                     //Create a new connection and assign values to it
                     result = mspin_tcp_createNewConnection(socketFD, cli_addr.sin_addr.s_addr);
                     if (MSPIN_SUCCESS == result)
                     {
                         //Socket connection accepted and new connection structure created
                         mspin_log_printLn(eMspinVerbosityInfo,
                                 "%s(listener=%p) Connection from '%s' (len=%d) accepted on port=%d with connID=%d",
                                 __FUNCTION__, pListener, ipAddr ? ipAddr : "n/a", ipAddr ? strlen(ipAddr) : 0,
                                 pListener->port, socketFD);
                     }
                     else
                     {
                         //No connection found. Reason can be either no connections or max number received
                         MSPIN_TCP_CONNECTION_END_REASON reason = MSPIN_TCP_CONNECTION_MAX_CONNECTIONS_REACHED;
                         if (MSPIN_ERROR_NOT_FOUND == result)
                         {
                             mspin_log_printLn(eMspinVerbosityError,
                                     "%s(listener=%p) ERROR: Max number connections reached! Reject from '%s' on port=%d with connID=%d",
                                     __FUNCTION__, pListener, ipAddr ? ipAddr : "n/a", pListener->port, socketFD);
                         }
                         else
                         {
                             reason = MSPIN_TCP_CONNECTION_OTHER_ERROR;
                             mspin_log_printLn(eMspinVerbosityError,
                                     "%s(listener=%p) ERROR: No TCP connection context! Reject from '%s' on port=%d with connID=%d",
                                     __FUNCTION__, pListener, ipAddr ? ipAddr : "n/a", pListener->port, socketFD);
                         }

                         close(socketFD);
                         socketFD = -1;
                         if (pListener->onClose)
                         {
                             pListener->onClose(-1, reason, (const U8*)ipAddr, pListener->onListenForConnectionsContext); //invalid connection ID used here to avoid problems
                         }
                     }
                 }
                 else
                 {
                     mspin_log_printLn(eMspinVerbosityInfo,
                             "%s(listener=%p) Ignore connection from '%s' (len=%d) on port=%d with connID=%d because listener is already stopped",
                             __FUNCTION__, pListener, ipAddr ? ipAddr : "n/a", ipAddr ? strlen(ipAddr) : 0, pListener->port, socketFD);

                     close(socketFD);
                     socketFD = -1;
                     //Do not call 'onClose' callback because listener is already stopped
                 }
             }
        }
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(listener=%p) leaving with fd=%d",
            __FUNCTION__, pListener, socketFD);

    return socketFD;
}

static bool mspin_tcp_confirmConnection(mspin_tcp_ListenerContext_t *pListener, int fd)
{
    in_addr_t ipAddr = 0;
    char hostString[INET_ADDRSTRLEN] = "";

    if (!pListener)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(listener=%p, fd=%d) FATAL ERROR: Context is NULL",
                __FUNCTION__, pListener, fd);
        return false;
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(listener=%p, fd=%d) entered", __FUNCTION__, pListener, fd);

    //Get IP address from connection
    ipAddr = mspin_tcp_getIPAddr(fd);

    if (0 == ipAddr)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(listener=%p, fd=%d) ERROR: Connection not found",
                __FUNCTION__, pListener, fd);
        return false;
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(listener=%p, fd=%d) after getting ipAddr=%u (=network address structure)",
            __FUNCTION__, pListener, fd, (unsigned int)ipAddr);

    if (!inet_ntop(AF_INET, &(ipAddr), hostString, INET_ADDRSTRLEN))
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(listener=%p, fd=%d) ERROR: Failed to convert IP address to string",
                __FUNCTION__, pListener, fd);

        return false;
    }

    if (pListener->onAccept)
    {
        if (false == pListener->onAccept(fd, (const U8*)hostString, pListener->onListenForConnectionsContext))
        {
            mspin_log_printLn(eMspinVerbosityInfo,
                    "%s(listener=%p, fd=%d) Connection from '%s' on port=%d rejected",
                    __FUNCTION__, pListener, fd, hostString, pListener->port);

            //Close connection
            (void)mspin_tcp_closeConnection(fd);
            mspin_tcp_signalConnectionClosed(fd, ipAddr, MSPIN_TCP_CONNECTION_REJECTED_BY_USER);

            return false;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s(listener=%p, fd=%d) Connection from '%s' on port=%d confirmed",
                    __FUNCTION__, pListener, fd, hostString, pListener->port);

            //Update connection status
            (void)mspin_tcp_setAccepted(fd);

            //Signal that we have a new socket accepted
            mspin_tcp_signalNewConnection();
        }
    }
    else
    {
        //No accept callback registered. This is not allowed. Therefore reject everything
        mspin_log_printLn(eMspinVerbosityWarn,
                "%s(listener=%p, fd=%d) WARNING: No acceptCB registered -> reject from '%s' on port=%d ",
                __FUNCTION__, pListener, fd, hostString, pListener->port);

        //Close connection
        (void)mspin_tcp_closeConnection(fd);
        mspin_tcp_signalConnectionClosed(fd, ipAddr, MSPIN_TCP_CONNECTION_OTHER_ERROR);

        return false;
    }

    return true;
}

static void* mspin_tcp_listenThread(void* exinf)
{
    int fd = -1;
    mspin_tcp_ListenerContext_t *pListener = (mspin_tcp_ListenerContext_t*)exinf;
    if (!pListener)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(exinf=%p) FATAL ERROR: TCP handle is NULL", __FUNCTION__, exinf);

        pthread_exit(NULL);
        return NULL;
    }

    //Set thread name
    prctl(PR_SET_NAME, "mspin_tcpListen", 0, 0, 0);

    //Accept incoming connection requests
    while (!pListener->quitListener && (pListener->listenFD > -1))
    {
        fd = mspin_tcp_accept(pListener);
        if (fd > -1)
        {
#ifdef MSPIN_TCP_USE_KEEPALIVE
            mspin_tcp_enableKeepAlive(fd);
#else
            mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) Keep-Alive is disabled per compile flag",
                    __FUNCTION__, exinf);
#endif //MSPIN_TCP_USE_KEEPALIVE

            (void)mspin_tcp_confirmConnection(pListener, fd); //ignore result. It does not matter here
        }
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) terminate thread", __FUNCTION__, exinf);

    pthread_exit(exinf);
    return NULL;
}

static S32 mspin_tcp_startListenThread(mspin_tcp_ListenerContext_t *pContext)
{
    pthread_attr_t attr;
    S32 rc = -1;

    memset(&attr, 0, sizeof(pthread_attr_t));

    rc = pthread_attr_init(&attr);
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to initialize thread attributes", __FUNCTION__);
        return rc;
    }

    rc = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to set detach state", __FUNCTION__);
        return false;
    }

    pContext->quitListener = FALSE;

    rc = pthread_create(&pContext->listenThreadID, &attr, mspin_tcp_listenThread, (void*)pContext);
    if (0 == rc)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s() using thread id=%d", __FUNCTION__, pContext->listenThreadID);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to create thread", __FUNCTION__);
        pContext->listenThreadID = 0;
    }

    (void)pthread_attr_destroy(&attr);

    return rc;
}

static void* mspin_tls_listenThread(void* exinf)
{
    int fd = -1;
    mspin_tls_accept_result_t tlsAcceptResult = MSPIN_TLS_SETUP_ERROR;
    MSPIN_TCP_CONNECTION_END_REASON reason = MSPIN_TCP_CONNECTION_REMOTE_END_HANG_UP;
    in_addr_t ipAddr = 0;
    char hostString[INET_ADDRSTRLEN] = "";

    mspin_tcp_ListenerContext_t *pListener = (mspin_tcp_ListenerContext_t*)exinf;
    if (!pListener)
    {
        mspin_log_printLn(eMspinVerbosityFatal, "%s(exinf=%p) FATAL ERROR: TCP handle is NULL", __FUNCTION__, exinf);

        pthread_exit(NULL);
        return NULL;
    }

    //Set thread name
    prctl(PR_SET_NAME, "mspin_tlsListen", 0, 0, 0);

    //Accept incoming connection requests
    while (!pListener->quitListener && (pListener->listenFD > -1))
    {
        fd = mspin_tcp_accept(pListener);
        if ((fd > -1) && (pListener->pTLSParameter))
        {
            mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) socket=%d accepted", __FUNCTION__, exinf, fd);

            ipAddr = mspin_tcp_getIPAddr(fd);

            tlsAcceptResult = mspin_tls_acceptConnection(fd, pListener->pTLSParameter->pSSLContext, pListener->pTLSParameter->verifyClient);
            if (MSPIN_TLS_CONNECTED == tlsAcceptResult)
            {
                (void)mspin_tcp_confirmConnection(pListener, fd); //ignore result. It does not matter here
            }
            else
            {
                inet_ntop(AF_INET, &(ipAddr), hostString, INET_ADDRSTRLEN);

                mspin_log_printLn(eMspinVerbosityError, "%s(exinf=%p) ERROR: TLS connection to %s not accepted",
                        __FUNCTION__, exinf, hostString);

                switch (tlsAcceptResult)
                {
                    case MSPIN_TLS_ACCEPT_ERROR:
                        reason = MSPIN_TCP_CONNECTION_TLS_ACCEPT_ERROR;
                        break;
                    case MSPIN_TLS_VERIFICATION_ERROR:
                        reason = MSPIN_TCP_CONNECTION_TLS_CLIENT_VERIFICATION_FAILED;
                        break;
                    case MSPIN_TLS_NO_CLIENT_CERT:
                        reason = MSPIN_TCP_CONNECTION_TLS_NO_CLIENT_CERT;
                        break;
                    case MSPIN_TLS_CONNECTION_ERROR:
                        reason = MSPIN_TCP_CONNECTION_SOCKET_ERROR;
                        break;
                    case MSPIN_TLS_SETUP_ERROR:
                    default:
                        reason = MSPIN_TCP_CONNECTION_REMOTE_END_HANG_UP;
                        break;
                }

                //Issue onClose callback
                if (pListener->onClose)
                {
                    pListener->onClose(fd, reason, (const U8*)hostString, pListener->onListenForConnectionsContext);
                }
            }
            mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) socket=%d done", __FUNCTION__, exinf, fd);
        }
        else if (!pListener->pTLSParameter)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(exinf=%p) ERROR: TLS parameter are NULL", __FUNCTION__, exinf);
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(exinf=%p) ERROR: mspin_tcp_accept returned with fd=%d",
                    __FUNCTION__, exinf, fd);
        }
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(exinf=%p) terminate thread", __FUNCTION__, exinf);

    pthread_exit(exinf);
    return NULL;
}

static S32 mspin_tls_startListenThread(mspin_tcp_ListenerContext_t *pContext)
{
    pthread_attr_t attr;
    S32 rc = -1;

    memset(&attr, 0, sizeof(pthread_attr_t));

    rc = pthread_attr_init(&attr);
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to inititialize thread attributes", __FUNCTION__);
        return rc;
    }

    rc = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (0 != rc)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to set detach state", __FUNCTION__);
        return false;
    }

    pContext->quitListener = FALSE;

    rc = pthread_create(&pContext->listenThreadID, &attr, mspin_tls_listenThread, (void*)pContext);
    if (0 == rc)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s() using thread id=%d", __FUNCTION__, pContext->listenThreadID);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to create thread", __FUNCTION__);
        pContext->listenThreadID = 0;
    }

    (void)pthread_attr_destroy(&attr);

    return rc;
}

static void mspin_tcp_joinThreads(void)
{
    void* status = NULL;

    if (gpTCPListenerContext)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s() flag threads to quit and closing listen socket...", __FUNCTION__);

        //Mark listener and watch thread to stop
        gpTCPListenerContext->quitListener = TRUE;

        //Close listening socket
        close(gpTCPListenerContext->listenFD);
        gpTCPListenerContext->listenFD = -1;

        mspin_log_printLn(eMspinVerbosityDebug, "%s() joining listen thread...", __FUNCTION__);

        if (0 != gpTCPListenerContext->listenThreadID)
        {
            (void)pthread_join(gpTCPListenerContext->listenThreadID, &status);
            gpTCPListenerContext->listenThreadID = 0;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityWarn, "%s() WARNING: No listen thread", __FUNCTION__);
        }

        //Reset values (socketFD will not be reset here because a mySPIN could still be state)
        gpTCPListenerContext->port = 0;
        gpTCPListenerContext->onAccept = NULL;
        gpTCPListenerContext->onClose = NULL;
        gpTCPListenerContext->onListenForConnectionsContext = NULL;

        mspin_log_printLn(eMspinVerbosityDebug, "%s() done", __FUNCTION__);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn, "%s() WARNING: No listener context found", __FUNCTION__);
    }
}

static mspin_tcp_ListenerContext_t* mspin_tcp_createListener(void)
{
    mspin_tcp_ListenerContext_t* pListener = NULL;

    pListener = malloc(sizeof(mspin_tcp_ListenerContext_t));

    if (!pListener)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to allocate memory for listener context",
                __FUNCTION__);
        return NULL;
    }

    memset(pListener, 0, sizeof(mspin_tcp_ListenerContext_t));

    mspin_log_printLn(eMspinVerbosityDebug, "%s() listener context=%p created", __FUNCTION__, pListener);

    return pListener;
}

static void mspin_tcp_deleteListener(mspin_tcp_ListenerContext_t** ppListener)
{
    mspin_tcp_ListenerContext_t* pListener = *ppListener;

    if (pListener)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(listener=%p) deleting values...",
                __FUNCTION__, pListener);

        //Additional steps
        if (pListener->pTLSParameter)
        {
            mspin_tls_deleteContext(&(pListener->pTLSParameter)); //sets it to NULL
        }

        mspin_log_printLn(eMspinVerbosityDebug, "%s(listener=%p) deleting context...",
                __FUNCTION__, pListener);

        free(pListener);
        *ppListener = NULL;

        mspin_log_printLn(eMspinVerbosityDebug, "%s(listener=%p) listener context deleted",
                __FUNCTION__, pListener);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityWarn, "%s(listener=%p) WARNING: Listener context is NULL, nothing to do",
                __FUNCTION__, pListener);
    }
}

//Create socket and bind it to local address and given port
static int mspin_tcp_listen(S32 tcpPort)
{
    int socketFD = -1;
    struct sockaddr_in serv_addr = {0};

#ifdef MSPIN_TCP_SOCKET_ADDR_REUSE
    int on = 1;
#endif //MSPIN_TCP_SOCKET_ADDR_REUSE

    //Create socket
    socketFD = socket(AF_INET, SOCK_STREAM, 0);
    if (-1 == socketFD)
    {
       mspin_log_printLn(eMspinVerbosityError, "%s(port=%d) ERROR: Failed to create the socket with '%s'(%d)",
               __FUNCTION__, tcpPort, strerror(errno), errno);
       return -1;
    }

#ifdef MSPIN_TCP_SOCKET_ADDR_REUSE
    //Enable socket option 'SO_REUSEADDR' to allow application to reuse the socket even if the socket is still in use.
    // This can happen when the application crashed but should not occur in normal operation.
    if (setsockopt(socketFD, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on)) < 0)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(port=%d) ERROR: Failed to setsockopt(SO_REUSEADDR) for fd=%d with '%s'(%d)",
               __FUNCTION__, tcpPort, socketFD, strerror(errno), errno);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(port=%d) setsockopt(SO_REUSEADDR) enabled for fd=%d",
               __FUNCTION__, tcpPort, socketFD);
    }
#endif //#ifdef MSPIN_TCP_SOCKET_ADDR_REUSE

    //Bind socket
    serv_addr.sin_family = AF_INET;
    serv_addr.sin_addr.s_addr = INADDR_ANY;
    serv_addr.sin_port = htons(tcpPort);
    /* PRQA: Lint Message 64: accept type mismatch */
    /*lint -save -e64*/
    if(-1 == bind(socketFD, (struct sockaddr *)&serv_addr, sizeof(serv_addr)))
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(port=%d) ERROR: Failed to bind the socket with '%s'(%d)",
              __FUNCTION__, tcpPort, strerror(errno), errno);

        close(socketFD);
        return -1;
    }
    /*lint -restore*/

    //Listen for incoming requests
    if (-1 == listen(socketFD, MSPIN_TCP_LISTEN_BACKLOG))
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(port=%d) ERROR: Failed to listen on socket with '%s'(%d)",
              __FUNCTION__, tcpPort, strerror(errno), errno);

        close(socketFD);
        return -1;
    }

    return(socketFD);
}

MSPIN_ERROR mspin_tcp_startListener(S32 tcpPort, MSPIN_TLS_CONFIGURATION_t* pTLSConfig,
        MSPIN_OnAcceptIncomingConnection acceptCB, MSPIN_OnConnectionClosed closedCB, void* callbackContext)
{
    int socketFD = -1;
    MSPIN_ERROR result = MSPIN_SUCCESS;

    //Initialize listener context
    if (!gpTCPListenerContext)
    {
        gpTCPListenerContext = mspin_tcp_createListener();

        if (!gpTCPListenerContext)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(port=%d) ERROR: Failed to create server context",
                   __FUNCTION__, tcpPort);
            return MSPIN_ERROR_GENERAL;
        }
    }
    else
    {
        //Check if still active (determine based on port. Port will be reset in mspin_tcp_joinThreads)
        if (gpTCPListenerContext->port > 0)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(port=%d) ERROR: Listener is still active. Stop it first",
                    __FUNCTION__, tcpPort);
            return MSPIN_ERROR_ALREADY_RUNNING;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityWarn, "%s(port=%d) WARNING: Server context is already present -> reuse",
                    __FUNCTION__, tcpPort);
        }
    }

    //Set initial values
    gpTCPListenerContext->listenFD = -1; //initialize socket FD with -1

    if (gpTCPListenerContext->pTLSParameter)
    {
        mspin_log_printLn(eMspinVerbosityWarn, "%s(port=%d) WARNING: TLS context is still present -> delete first",
                __FUNCTION__, tcpPort);

        mspin_tls_deleteContext(&(gpTCPListenerContext->pTLSParameter));
    }

    if (pTLSConfig)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(port=%d) using TLS configuration", __FUNCTION__, tcpPort);

        //Initialize SSL/TLS
        mspin_tls_initTLS();

        //Create TLS context
        gpTCPListenerContext->pTLSParameter = mspin_tls_createContext(); //only in TLS case required

        //Configure certificate etc. filenames for TLS
        mspin_tls_setParameter(gpTCPListenerContext->pTLSParameter, pTLSConfig);

        //Create TLS server configuration
        result = mspin_tls_createSSLContext(gpTCPListenerContext->pTLSParameter);
        if (MSPIN_SUCCESS != result)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(port=%d) ERROR: Failed to create TLS context",
                   __FUNCTION__, tcpPort);

            mspin_tcp_deleteListener(&gpTCPListenerContext);
            return result;
        }

        //Configure TLS server context
        result = mspin_tls_configureSSLContext(gpTCPListenerContext->pTLSParameter);
        if (MSPIN_SUCCESS != result)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(port=%d) ERROR: Failed to configure TLS context",
                   __FUNCTION__, tcpPort);

            mspin_tcp_deleteListener(&gpTCPListenerContext);
            return result;
        }

        gpTCPListenerContext->secured = TRUE;
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(port=%d) using TCP/IP configuration", __FUNCTION__, tcpPort);
    }

    //Set some parameters before starting watch thread
    gpTCPListenerContext->onAccept = acceptCB;
    gpTCPListenerContext->onClose = closedCB;
    gpTCPListenerContext->onListenForConnectionsContext = callbackContext;

    //Start watching
    if (mspin_tcp_startWatching() != MSPIN_SUCCESS)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(port=%d) ERROR: startWatching failed",
              __FUNCTION__, tcpPort);

        close(socketFD);
        mspin_tcp_deleteListener(&gpTCPListenerContext);
        return MSPIN_ERROR_GENERAL;
    }

    //Create socket, bind it to local address and specified port and start listening
    socketFD = mspin_tcp_listen(tcpPort);
    if (-1 == socketFD)
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(port=%d) ERROR: Failed to create socket",
               __FUNCTION__, tcpPort);

        mspin_tcp_deleteListener(&gpTCPListenerContext);
        return MSPIN_ERROR_GENERAL;
    }

    //Now set the remaining parameters
    gpTCPListenerContext->listenFD = socketFD;
    gpTCPListenerContext->port = tcpPort;

    //Start thread to accept incoming connections
    if (pTLSConfig)
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(port=%d) start TLS listen thread", __FUNCTION__, tcpPort);

        if (0 != mspin_tls_startListenThread(gpTCPListenerContext))
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(port=%d) ERROR: Failed to start watch thread",
                  __FUNCTION__, tcpPort);

            close(socketFD);
            mspin_tcp_stopWatching(); //join watch thread

            mspin_tcp_deleteListener(&gpTCPListenerContext);
            return MSPIN_ERROR_GENERAL;
        }
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityDebug, "%s(port=%d) start TCP/IP listen thread", __FUNCTION__, tcpPort);

        if (0 != mspin_tcp_startListenThread(gpTCPListenerContext))
        {
           mspin_log_printLn(eMspinVerbosityError, "%s(port=%d) ERROR: Failed to start watch thread",
                   __FUNCTION__, tcpPort);

           close(socketFD);
           mspin_tcp_stopWatching(); //join watch thread

           mspin_tcp_deleteListener(&gpTCPListenerContext);
           return MSPIN_ERROR_GENERAL;
        }
    }

    mspin_log_printLn(eMspinVerbosityDebug, "%s(port=%d) TCP listen thread started", __FUNCTION__, tcpPort);

    return MSPIN_SUCCESS;
}

MSPIN_ERROR mspin_tcp_stopListener(BOOL secured)
{
    mspin_log_printLn(eMspinVerbosityDebug, "%s(TLS=%s) entered", __FUNCTION__, secured ? "TRUE" : "FALSE");

    if (gpTCPListenerContext)
    {
        if (gpTCPListenerContext->secured != secured)
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(TLS=%s) ERROR: There is only a %s listener running -> stop correct one",
                    __FUNCTION__, secured ? "TRUE" : "FALSE", gpTCPListenerContext->secured ? "TLS" : "TCP/IP");
            return MSPIN_ERROR_NOT_FOUND;
        }

        //Join listener thread
        mspin_tcp_joinThreads();

        mspin_log_printLn(eMspinVerbosityDebug, "%s(TLS=%s) listener stopped. Stop watch thread now",
                __FUNCTION__, gpTCPListenerContext->secured ? "TRUE" : "FALSE");

        //Join watch thread
        mspin_tcp_stopWatching();

        mspin_log_printLn(eMspinVerbosityDebug, "%s(TLS=%s) Watch thread stopped. Delete listener now",
                __FUNCTION__, gpTCPListenerContext->secured ? "TRUE" : "FALSE");

        //Delete listener
        mspin_tcp_deleteListener(&gpTCPListenerContext);
    }
    else
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(TLS=%s) ERROR: No listener present",
                __FUNCTION__, secured ? "TRUE" : "FALSE");

        //Join watch thread
        mspin_tcp_stopWatching();
    }

    return MSPIN_SUCCESS;
}

void mspin_tcp_signalConnectionClosed(int socketFD, in_addr_t ipAddr, MSPIN_TCP_CONNECTION_END_REASON reason)
{
    char hostString[INET_ADDRSTRLEN] = "";

    if (gpTCPListenerContext && gpTCPListenerContext->onClose)
    {
        inet_ntop(AF_INET, &(ipAddr), hostString, INET_ADDRSTRLEN);
        gpTCPListenerContext->onClose(socketFD, reason, (const U8*)hostString, gpTCPListenerContext->onListenForConnectionsContext);
    }
    else if (gpTCPListenerContext)
    {
        mspin_log_printLn(eMspinVerbosityInfo,
                "%s(socketFD=%d, ipAddr='%s', reason=%s) onClose callback is not registered",
                __FUNCTION__, socketFD, hostString, mspin_tcp_getReasonName(reason));
    }
}
